% Matlab code for the simulation of the minimal physical model presented in "Persistent cell migration emerges from a coupling between protrusion dynamics and polarized trafficking by Vaidiulyte et al.
% The code should be run once to produce a movie of a synthetic migrating
% cell
% Written by M. Coppey, 03-25-2021

%% parameters of the model
clear all
tt = 300; % total duration of the simulation [1000]
L = 101; % number of markers along the cell contour
pp = 0.2; % probability of protrusion per unit of time [0.5]
ptime = 30; % intensity of protrusions in the fake cell
simgaNG = 20; % spatial extent of the protrusion bias toward the polarity axis [0.8]
speedNG = 0.5; % speed of the polarity axis reorientation
feed1 = 1; % feedback polarity axis --> protrusion on =1, off =0
feed2 = 1; % feedback protrusion --> polarity axis on =1, off =3
pG = 0.5; % probability to make a protrusion in front of the polarity axis [0.8]
NGforce = 0.8; % bias of the polarity axis towards protrusions% transfer matrix [0.8]
Ttot = 100;

% create transfer matrix
x = -50:50;
H = zeros(Ttot+1,101); % H transfer function
c = 0.5; % speed
f1 = 0.1; % fade factor 1
f2 = 0.1; % fade factor 2
f3 = 0.2; % fade factor 3
for t=0:Ttot
    % initial lateral inhibition:
    gauss1 = -0.01*exp(-f1*t);
    % traveling positive waves:
    gauss2 = (0.5.*normpdf(x,c.*t,5)+0.5.*normpdf(x,-c.*t,5))*exp(-f2*t);
    % central long lasting inhibition
    gauss3 = -(0.1.*normpdf(x,0,3).*normpdf(t,35,25).*sqrt(2*pi).*20);
    % central very peaked positve function
    gauss4 = 0.1.*normpdf(x,0,1).*normpdf(t,10,5).*sqrt(2*pi);
    h0 = gauss2+gauss1+gauss3+gauss4;
    H(t+1,:)=h0;
end
% ensure zero mean
int = sum(sum(H));
H = H-int./numel(H);
maxt=max(max(H));
Hf = flipud(H);
% convolve with signal
signal= zeros(Ttot+1,101); % H transfer function
f1 = 0.1; % time scale to appear
f2 = 10; % duration
f3 = 0.3; % fading time scale
for t=0:Ttot
    if t<f2
        signal(t+1,:)= (1-exp(-f1.*t)).*normpdf(x,0,5);
    else
        signal(t+1,:)= exp(-f3.*(t-f2)).* (1-exp(-f1.*t)).*normpdf(x,0,5);
    end
end
% ensure 1 mean
int = sum(sum(signal));
signal = signal-int./numel(signal);
M = conv2(signal,H);
M = M(1:Ttot+1,50:150);
H = M;
int = sum(sum(H));
H = H-int./numel(H);
% s simulate the whole morphodynamic map 
Mk = zeros(L,tt);
pos = [];
np = 0; % number of protrusions
NG = zeros(1,tt);
NGangle = zeros(1,tt);
% fake cell
[columnsInImage rowsInImage] = meshgrid(1:1000, 1:1000);
radius = 100;
bw = (rowsInImage - 500).^2 + (columnsInImage - 500).^2 <= radius.^2;
se=strel('disk', 1);
numPoints = 101;
boundbw=bwboundaries(bw);
boundbw=boundbw{1};
boundaryLength=size(boundbw,1)-1;
indexCol=[0:(numPoints/boundaryLength):numPoints];
edges(:,1)=interp1(indexCol,boundbw(:,1),0.5:numPoints);
edges(:,2)=interp1(indexCol,boundbw(:,2),0.5:numPoints);
mass = polyarea(edges(:,1),edges(:,2));
NGprev = 1;
ycp = 500;
xcp = 500;
flag=1;
figure(1)
% get the figure and axes handles
hFig = gcf;
hAx  = gca;
% set the figure to full screen
set(hFig,'units','normalized','outerposition',[0 0.5 0.15 0.5]);
% set the axes to full screen
set(hAx,'Unit','normalized','Position',[0 0 1 1]);
for time=1:tt % main loop
    % find the NG marker
    NGangle(time) = NGprev;
    xc(time)=xcp;
    yc(time)=ycp;
    XY1 = [xc(time) yc(time) xc(time)+1000*cos(NGangle(time)) yc(time)+1000*sin(NGangle(time))] ;
    XY2 = [];
    for k=1:L-1
        XY2 = [XY2;edges(k,1) edges(k,2) edges(k+1,1) edges(k+1,2)];
    end
    XY2 = [XY2;edges(L,1) edges(L,2) edges(1,1) edges(1,2)];
    out = lineSegmentIntersect(XY1,XY2);
    [~,i] = find(out.intAdjacencyMatrix==1);
    if length(i)>1
        i = i(1);
    end
    if isempty(i)
        i = iprev;
    end
    xint = out.intMatrixX(i);
    yint = out.intMatrixY(i);
    if i==L
        frac = max([(xint-edges(i,1))/(edges(1,1)-edges(i,1))  (yint-edges(i,2))/(edges(1,2)-edges(i,2))]);
        NG(time) = 1+frac;
    else
        frac = max([(xint-edges(i,1))/(edges(i+1,1)-edges(i,1))  (yint-edges(i,2))/(edges(i+1,2)-edges(i,2))]);
        NG(time) = i+frac;
    end
    iprev = i;
    if rand<pp % compute if a new protrusion is made
        if feed1==0
            posloc = ceil(L*rand); % feedback off
        else
            posloc = round(normrnd(NG(time),simgaNG)); % feedback on
        end
        if posloc>L
            posloc=posloc-L;
        elseif posloc<1
            posloc = L+posloc;
        end
        pos = [pos;posloc,time];
        np =np+1;
    end
    for k=1:np
        pro = pos(k,:); % current protrusion
        if time-pro(2)<Ttot+1 %time of the protrusion
            locM = H(time-pro(2)+1,:);
            locM = circshift(locM,50+pro(1));
            Mk(:,time) = Mk(:,time)+locM';
        end
    end % compute the map given the protrusions
    if feed2==1 % update polarity axis positioning
        dF = 0;
        for k=1:L
            if (abs(k-NG(time))>=L-abs(NG(time)-k))
                distpt = L-abs(NG(time)-k);
                if Mk(k,time)>=0
                    dF(k) = -sign(k-NG(time))*Mk(k,time);
                end
            else
                distpt = abs(k-NG(time));
                if Mk(k,time)>=0
                    dF(k) = sign(k-NG(time))*Mk(k,time);
                end
            end
        end
        NG(time) = NG(time)+round(NGforce*sum(dF));
        if NG(time)>L
            NG(time)=NG(time)-L;
        elseif NG(time)<1
            NG(time) = L+NG(time);
        end
    end
    % find the polarity angle given the NG marker
    i = floor(NG(time));
    if i==L
        frac = NG(time)-L;
        xint = edges(i,1)+frac*(edges(1,1)-edges(i,1));
        yint =  edges(i,2)+frac*(edges(1,2)-edges(i,2));
    elseif i==0
        frac = NG(time);
        xint = edges(L,1)+frac*(edges(L,1)-edges(1,1));
        yint =  edges(L,2)+frac*(edges(L,2)-edges(1,2));
    else
        frac = NG(time)-i;
        xint = edges(i,1)+frac*(edges(i+1,1)-edges(i,1));
        yint =  edges(i,2)+frac*(edges(i+1,2)-edges(i,2));
    end
    NGangle(time) = atan2(yint-yc(time),xint-xc(time));
    NGprev = NGangle(time);
    % apply morphodynamics
    pt = edges;
    npt = length(pt);
    x = zeros(npt,1); %point to find
    y = zeros(npt,1); %point to find
    for i=1:npt
        a = ptime*Mk(i,time);
        %a = 100*H(1,i);
        if i==1
            x1 = pt(end,1); y1 = pt(end,2); x2 = pt(i+1,1); y2 = pt(i+1,2); x0 = pt(i,1); y0 = pt(i,2);
        elseif i==npt
            x1 = pt(i-1,1); y1 = pt(i-1,2); x2 = pt(1,1); y2 = pt(1,2); x0 = pt(i,1); y0 = pt(i,2);
        else
            x1 = pt(i-1,1); y1 = pt(i-1,2); x2 = pt(i+1,1); y2 = pt(i+1,2); x0 = pt(i,1); y0 = pt(i,2);
        end
        if y1==y2
            xp = x0;
            xm = x0;
            yp = y0+a;
            ym = y0-a;
        else
            b = (x2-x1)/(y2-y1);
            xp = x0+sqrt(a^2/(1+2*abs(b)+b^2));
            xm = x0-sqrt(a^2/(1+2*abs(b)+b^2));
            yp = y0-b*(xp-x0);
            ym = y0-b*(xm-x0);
        end
        if a>0
            if xp>=1000|yp>=1000|xp<=0|yp<=0
                flag=0;
            else
                if bw(round(xp),round(yp))==0
                    x(i) = xp;
                    y(i) = yp;
                else
                    x(i) = xm;
                    y(i) = ym;
                end
            end
        else
            if bw(round(xp),round(yp))==1
                x(i) = xp;
                y(i) = yp;
            else
                x(i) = xm;
                y(i) = ym;
            end
        end
        if flag==0;
            break
        end
    end
    if flag==0;
        break
    end
    % optional plot
    %
    %                 imshow(bw)
    %                 hold on
    %                 plot(edges(:,2),edges(:,1),'ob')
    %               hold on
    %                 plot(y,x,'.r')
    q = curvspace([x,y],numPoints);
    edges(:,1) = q(:,1);
    edges(:,2) = q(:,2);
    % ensure cell mass conservation
    massn = polyarea(edges(:,1),edges(:,2));
    factmass = mass./massn;
    pgon = polyshape(edges(:,2),edges(:,1),'Simplify',false);
    [yc(time),xc(time)] = centroid(pgon);
    xcp = xc(time);
    ycp = yc(time);
    polyout = scale(pgon,factmass,[yc(time),xc(time)]);
    edges(:,1) = polyout.Vertices(:,2);
    edges(:,2) = polyout.Vertices(:,1);
    % alignement to the lab referential
    cxe = edges(:,1)-xcp;
    cye = edges(:,2)-ycp;
    indneg = find(cxe<0);
    [~,If] = min(abs(cye(indneg)));
    phase = indneg(If);
    tot = length(edges);
    temp = zeros(size(edges));
    temp(1:tot-phase,:) = edges(phase+1:end,:);
    temp(tot-phase+1:end,:) = edges(1:phase,:);
    edges = temp;
    bw = poly2mask(edges(:,2), edges(:,1), 1000, 1000);
    subplot(4,1,1:3)
    imshow(bw,'initialmagnification',200)
    hold on
    quiver(yc(time),xc(time),50.*sin(NGangle(time)),50.*cos(NGangle(time)),'MaxHeadSize',1,'LineWidth',2,'color','r')
    hold on
    plot(yc(1:time),xc(1:time),'color','c','linewidth',1)
    subplot(4,1,4)
    imshow(Mk,[])
    colormap(gca,'jet')
    hold on
    plot((1:tt),NG,'.k')
    drawnow
    Mov(time) = getframe(gcf);
end
% myVideo = VideoWriter('movie.avi', 'Uncompressed AVI');
myVideo = VideoWriter('movie', 'MPEG-4');
myVideo.FrameRate = 24;  % Default 30
open(myVideo);
writeVideo(myVideo, Mov);
close(myVideo);

